
/* ====================================================================
   Licensed to the Apache Software Foundation (ASF) under one or more
   contributor license agreements.  See the NOTICE file distributed with
   this work for Additional information regarding copyright ownership.
   The ASF licenses this file to You under the Apache License, Version 2.0
   (the "License"); you may not use this file except in compliance with
   the License.  You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License Is distributed on an "AS Is" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
==================================================================== */



namespace NPOI.HDF.Model.HDFTypes
{
    using System;
    using NPOI.HDF.Model.HDFTypes.Definitions;
    using NPOI.Util;

    /**
     * Represents a document's stylesheet. A word documents formatting Is stored as
     * compressed styles that are based on styles contained in the stylesheet. This
     * class also Contains static utility functions to uncompress different
     * formatting properties.
     *
     * @author Ryan Ackley
     */

    public class StyleSheet : HDFType
    {

        private const int NIL_STYLE = 4095;
        private const int PAP_TYPE = 1;
        private const int CHP_TYPE = 2;
        private const int SEP_TYPE = 4;
        private const int TAP_TYPE = 5;
        //Vector _styleDescriptions;
        StyleDescription _nilStyle = new StyleDescription();
        StyleDescription[] _styleDescriptions;

        /**
         * StyleSheet constructor. Loads a document's stylesheet information,
         *
         * @param styleSheet A byte array containing a document's raw stylesheet
         *        info. Found by using FileInformationBlock.GetFcStshf() and
         *        FileInformationBLock.GetLcbStshf()
         */
        public StyleSheet(byte[] styleSheet)
        {
            int stshiLength = LittleEndian.GetShort(styleSheet, 0);
            int stdCount = LittleEndian.GetShort(styleSheet, 2);
            int baseLength = LittleEndian.GetShort(styleSheet, 4);
            int[] rgftc = new int[3];

            rgftc[0] = LittleEndian.GetInt(styleSheet, 14);
            rgftc[1] = LittleEndian.GetInt(styleSheet, 18);
            rgftc[2] = LittleEndian.GetInt(styleSheet, 22);

            int offset = 0;
            _styleDescriptions = new StyleDescription[stdCount];
            for (int x = 0; x < stdCount; x++)
            {
                int stdOffset = (2 + stshiLength) + offset;
                int stdSize = LittleEndian.GetShort(styleSheet, stdOffset);
                if (stdSize > 0)
                {
                    byte[] std = new byte[stdSize];

                    //get past the Size
                    stdOffset += 2;
                    System.Array.Copy(styleSheet, stdOffset, std, 0, stdSize);
                    StyleDescription aStyle = new StyleDescription(std, baseLength, true);

                    _styleDescriptions[x] = aStyle;
                }


                offset += stdSize + 2;

            }
            for (int x = 0; x < _styleDescriptions.Length; x++)
            {
                if (_styleDescriptions[x] != null)
                {
                    CreatePap(x);
                    CreateChp(x);
                }
            }
        }
        /**
         * Creates a PartagraphProperties object from a papx stored in the
         * StyleDescription at the index Istd in the StyleDescription array. The PAP
         * Is placed in the StyleDescription at Istd after its been Created. Not
         * every StyleDescription will contain a papx. In these cases this function
         * does nothing
         *
         * @param Istd The index of the StyleDescription to Create the
         *        ParagraphProperties  from (and also place the finished PAP in)
         */
        private void CreatePap(int istd)
        {
            StyleDescription sd = _styleDescriptions[istd];
            ParagraphProperties pap = sd.GetPAP();
            byte[] papx = sd.GetPAPX();
            int baseIndex = sd.GetBaseStyle();
            if (pap == null && papx != null)
            {
                ParagraphProperties parentPAP = _nilStyle.GetPAP();
                if (baseIndex != NIL_STYLE)
                {

                    parentPAP = _styleDescriptions[baseIndex].GetPAP();
                    if (parentPAP == null)
                    {
                        CreatePap(baseIndex);
                        parentPAP = _styleDescriptions[baseIndex].GetPAP();
                    }

                }

                pap = (ParagraphProperties)uncompressProperty(papx, parentPAP, this);
                sd.SetPAP(pap);
            }
        }
        /**
         * Creates a CharacterProperties object from a chpx stored in the
         * StyleDescription at the index Istd in the StyleDescription array. The
         * CharacterProperties object Is placed in the StyleDescription at Istd after
         * its been Created. Not every StyleDescription will contain a chpx. In these
         * cases this function does nothing.
         *
         * @param Istd The index of the StyleDescription to Create the
         *        CharacterProperties object from.
         */
        private void CreateChp(int istd)
        {
            StyleDescription sd = _styleDescriptions[istd];
            CharacterProperties chp = sd.GetCHP();
            byte[] chpx = sd.GetCHPX();
            int baseIndex = sd.GetBaseStyle();
            if (chp == null && chpx != null)
            {
                CharacterProperties parentCHP = _nilStyle.GetCHP();
                if (baseIndex != NIL_STYLE)
                {

                    parentCHP = _styleDescriptions[baseIndex].GetCHP();
                    if (parentCHP == null)
                    {
                        CreateChp(baseIndex);
                        parentCHP = _styleDescriptions[baseIndex].GetCHP();
                    }

                }

                chp = (CharacterProperties)uncompressProperty(chpx, parentCHP, this);
                sd.SetCHP(chp);
            }
        }

        /**
         * Gets the StyleDescription at index x.
         *
         * @param x the index of the desired StyleDescription.
         */
        public StyleDescription GetStyleDescription(int x)
        {
            return _styleDescriptions[x];
        }

        /**
         * Used in decompression of a chpx. This performs an operation defined by
         * a single sprm.
         *
         * @param oldCHP The base CharacterProperties.
         * @param newCHP The current CharacterProperties.
         * @param operand The operand defined by the sprm (See Word file format spec)
         * @param param The parameter defined by the sprm (See Word file format spec)
         * @param varParam The variable Length parameter defined by the sprm. (See
         *        Word file format spec)
         * @param grpprl The entire chpx that this operation Is a part of.
         * @param offset The offset in the grpprl of the next sprm
         * @param styleSheet The StyleSheet for this document.
         */
        static void DoCHPOperation(CharacterProperties oldCHP, CharacterProperties newCHP,
                                   int operand, int param,
                                   byte[] varParam, byte[] grpprl, int offset,
                                   StyleSheet styleSheet)
  {
      switch(operand)
      {
          case 0:
               newCHP.SetFRMarkDel(GetFlag(param));
               break;
          case 0x1:
               newCHP.SetFRMark(GetFlag(param));
               break;
          case 0x2:
               break;
          case 0x3:
               newCHP.SetFcPic(param);
               newCHP.SetFSpec(true);
               break;
          case 0x4:
               newCHP.SetIbstRMark((short)param);
               break;
          case 0x5:
               short[] dttmRMark = new short[2];
               dttmRMark[0] = LittleEndian.GetShort(grpprl, (offset - 4));
               dttmRMark[1] = LittleEndian.GetShort(grpprl, (offset - 2));
               newCHP.SetDttmRMark(dttmRMark);
               break;
          case 0x6:
               newCHP.SetFData(GetFlag(param));
               break;
          case 0x7:
               //don't care about this
               break;
          case 0x8:
               short chsDiff = (short)((param & 0xff0000) >> 8);
               newCHP.SetFChsDiff(GetFlag(chsDiff));
               newCHP.SetChse((short)(param & 0xffff));
               break;
          case 0x9:
               newCHP.SetFSpec(true);
               newCHP.SetFtcSym((short)LittleEndian.GetShort(varParam, 0));
               newCHP.SetXchSym((short)LittleEndian.GetShort(varParam, 2));
               break;
          case 0xa:
               newCHP.SetFOle2(GetFlag(param));
               break;
          case 0xb:
               //?
               break;
          case 0xc:
               newCHP.SetIcoHighlight((byte)param);
               newCHP.SetFHighlight(GetFlag(param));
               break;
          case 0xd:
               break;
          case 0xe:
               newCHP.SetFcObj(param);
               break;
          case 0xf:
               break;
          case 0x10:
               //?
               break;
          case 0x11:
               break;
          case 0x12:
               break;
          case 0x13:
               break;
          case 0x14:
               break;
          case 0x15:
               break;
          case 0x16:
               break;
          case 0x17:
               break;
          case 0x18:
               break;
          case 0x19:
               break;
          case 0x1a:
               break;
          case 0x1b:
               break;
          case 0x1c:
               break;
          case 0x1d:
               break;
          case 0x1e:
               break;
          case 0x1f:
               break;
          case 0x20:
               break;
          case 0x21:
               break;
          case 0x22:
               break;
          case 0x23:
               break;
          case 0x24:
               break;
          case 0x25:
               break;
          case 0x26:
               break;
          case 0x27:
               break;
          case 0x28:
               break;
          case 0x29:
               break;
          case 0x2a:
               break;
          case 0x2b:
               break;
          case 0x2c:
               break;
          case 0x2d:
               break;
          case 0x2e:
               break;
          case 0x2f:
               break;
          case 0x30:
               newCHP.SetIstd(param);
               break;
          case 0x31:
               //permutation vector for fast saves, who cares!
               break;
          case 0x32:
               newCHP.SetFBold(false);
               newCHP.SetFItalic(false);
               newCHP.SetFOutline(false);
               newCHP.SetFStrike(false);
               newCHP.SetFShadow(false);
               newCHP.SetFSmallCaps(false);
               newCHP.SetFCaps(false);
               newCHP.SetFVanish(false);
               newCHP.SetKul((byte)0);
               newCHP.SetIco((byte)0);
               break;
          case 0x33:
                   newCHP = (CharacterProperties)oldCHP.Clone();
               return;
          case 0x34:
               break;
          case 0x35:
               newCHP.SetFBold(GetCHPFlag((byte)param, oldCHP.IsFBold()));
               break;
          case 0x36:
               newCHP.SetFItalic(GetCHPFlag((byte)param, oldCHP.IsFItalic()));
               break;
          case 0x37:
               newCHP.SetFStrike(GetCHPFlag((byte)param, oldCHP.IsFStrike()));
               break;
          case 0x38:
               newCHP.SetFOutline(GetCHPFlag((byte)param, oldCHP.IsFOutline()));
               break;
          case 0x39:
               newCHP.SetFShadow(GetCHPFlag((byte)param, oldCHP.IsFShadow()));
               break;
          case 0x3a:
               newCHP.SetFSmallCaps(GetCHPFlag((byte)param, oldCHP.IsFSmallCaps()));
               break;
          case 0x3b:
               newCHP.SetFCaps(GetCHPFlag((byte)param, oldCHP.IsFCaps()));
               break;
          case 0x3c:
               newCHP.SetFVanish(GetCHPFlag((byte)param, oldCHP.IsFVanish()));
               break;
          case 0x3d:
               newCHP.SetFtcAscii((short)param);
               break;
          case 0x3e:
               newCHP.SetKul((byte)param);
               break;
          case 0x3f:
               int hps = param & 0xff;
               if(hps != 0)
               {
                  newCHP.SetHps(hps);
               }
               byte cInc = (byte)(((byte)(param & 0xfe00) >> 4) >> 1);
               if(cInc != 0)
               {
                  newCHP.SetHps(Math.Max(newCHP.GetHps() + (cInc * 2), 2));
               }
               byte hpsPos = (byte)((param & 0xff0000) >> 8);
               if(hpsPos != 0x80)
               {
                  newCHP.SetHpsPos(hpsPos);
               }
               bool fAdjust = (param & 0x0100) > 0;
               if(fAdjust && hpsPos != 128 && hpsPos != 0 && oldCHP.GetHpsPos() == 0)
               {
                  newCHP.SetHps(Math.Max(newCHP.GetHps() + (-2), 2));
               }
               if(fAdjust && hpsPos == 0 && oldCHP.GetHpsPos() != 0)
               {
                  newCHP.SetHps(Math.Max(newCHP.GetHps() + 2, 2));
               }
               break;
          case 0x40:
               newCHP.SetDxaSpace(param);
               break;
          case 0x41:
               newCHP.SetLidDefault((short)param);
               break;
          case 0x42:
               newCHP.SetIco((byte)param);
               break;
          case 0x43:
               newCHP.SetHps(param);
               break;
          case 0x44:
               byte hpsLvl = (byte)param;
               newCHP.SetHps(Math.Max(newCHP.GetHps() + (hpsLvl * 2), 2));
               break;
          case 0x45:
               newCHP.SetHpsPos((short)param);
               break;
          case 0x46:
               if(param != 0)
               {
                  if(oldCHP.GetHpsPos() == 0)
                  {
                      newCHP.SetHps(Math.Max(newCHP.GetHps() + (-2), 2));
                  }
               }
               else
               {
                  if(oldCHP.GetHpsPos() != 0)
                  {
                      newCHP.SetHps(Math.Max(newCHP.GetHps() + 2, 2));
                  }
               }
               break;
          case 0x47:
               CharacterProperties genCHP = new CharacterProperties();
               genCHP.SetFtcAscii(4);
               genCHP = (CharacterProperties)uncompressProperty(varParam, genCHP, styleSheet);
               CharacterProperties styleCHP = styleSheet.GetStyleDescription(oldCHP.GetBaseIstd()).GetCHP();
               if(genCHP.IsFBold() == newCHP.IsFBold())
               {
                  newCHP.SetFBold(styleCHP.IsFBold());
               }
               if(genCHP.IsFItalic() == newCHP.IsFItalic())
               {
                  newCHP.SetFItalic(styleCHP.IsFItalic());
               }
               if(genCHP.IsFSmallCaps() == newCHP.IsFSmallCaps())
               {
                  newCHP.SetFSmallCaps(styleCHP.IsFSmallCaps());
               }
               if(genCHP.IsFVanish() == newCHP.IsFVanish())
               {
                  newCHP.SetFVanish(styleCHP.IsFVanish());
               }
               if(genCHP.IsFStrike() == newCHP.IsFStrike())
               {
                  newCHP.SetFStrike(styleCHP.IsFStrike());
               }
               if(genCHP.IsFCaps() == newCHP.IsFCaps())
               {
                  newCHP.SetFCaps(styleCHP.IsFCaps());
               }
               if(genCHP.GetFtcAscii() == newCHP.GetFtcAscii())
               {
                  newCHP.SetFtcAscii(styleCHP.GetFtcAscii());
               }
               if(genCHP.GetFtcFE() == newCHP.GetFtcFE())
               {
                  newCHP.SetFtcFE(styleCHP.GetFtcFE());
               }
               if(genCHP.GetFtcOther() == newCHP.GetFtcOther())
               {
                  newCHP.SetFtcOther(styleCHP.GetFtcOther());
               }
               if(genCHP.GetHps() == newCHP.GetHps())
               {
                  newCHP.SetHps(styleCHP.GetHps());
               }
               if(genCHP.GetHpsPos() == newCHP.GetHpsPos())
               {
                  newCHP.SetHpsPos(styleCHP.GetHpsPos());
               }
               if(genCHP.GetKul() == newCHP.GetKul())
               {
                  newCHP.SetKul(styleCHP.GetKul());
               }
               if(genCHP.GetDxaSpace() == newCHP.GetDxaSpace())
               {
                  newCHP.SetDxaSpace(styleCHP.GetDxaSpace());
               }
               if(genCHP.GetIco() == newCHP.GetIco())
               {
                  newCHP.SetIco(styleCHP.GetIco());
               }
               if(genCHP.GetLidDefault() == newCHP.GetLidDefault())
               {
                  newCHP.SetLidDefault(styleCHP.GetLidDefault());
               }
               if(genCHP.GetLidFE() == newCHP.GetLidFE())
               {
                  newCHP.SetLidFE(styleCHP.GetLidFE());
               }
               break;
          case 0x48:
               newCHP.SetIss((byte)param);
               break;
          case 0x49:
               newCHP.SetHps(LittleEndian.GetShort(varParam, 0));
               break;
          case 0x4a:
               int increment = LittleEndian.GetShort(varParam, 0);
               newCHP.SetHps(Math.Max(newCHP.GetHps() + increment, 8));
               break;
          case 0x4b:
               newCHP.SetHpsKern(param);
               break;
          case 0x4c:
               DoCHPOperation(oldCHP, newCHP, 0x47, param, varParam, grpprl, offset, styleSheet);
               break;
          case 0x4d:
               float percentage = (float)param/100.0f;
               int Add = (int)((float)percentage * (float)newCHP.GetHps());
               newCHP.SetHps(newCHP.GetHps() + Add);
               break;
          case 0x4e:
               newCHP.SetYsr((byte)param);
               break;
          case 0x4f:
               newCHP.SetFtcAscii((short)param);
               break;
          case 0x50:
               newCHP.SetFtcFE((short)param);
               break;
          case 0x51:
               newCHP.SetFtcOther((short)param);
               break;
          case 0x52:
               break;
          case 0x53:
               newCHP.SetFDStrike(GetFlag(param));
               break;
          case 0x54:
               newCHP.SetFImprint(GetFlag(param));
               break;
          case 0x55:
               newCHP.SetFSpec(GetFlag(param));
               break;
          case 0x56:
               newCHP.SetFObj(GetFlag(param));
               break;
          case 0x57:
               newCHP.SetFPropMark(varParam[0]);
               newCHP.SetIbstPropRMark((short)LittleEndian.GetShort(varParam, 1));
               newCHP.SetDttmPropRMark(LittleEndian.GetInt(varParam, 3));
               break;
          case 0x58:
               newCHP.SetFEmboss(GetFlag(param));
               break;
          case 0x59:
               newCHP.SetSfxtText((byte)param);
               break;
          case 0x5a:
               break;
          case 0x5b:
               break;
          case 0x5c:
               break;
          case 0x5d:
               break;
          case 0x5e:
               break;
          case 0x5f:
               break;
          case 0x60:
               break;
          case 0x61:
               break;
          case 0x62:
               byte[] xstDispFldRMark = new byte[32];
               newCHP.SetFDispFldRMark(varParam[0]);
               newCHP.SetIbstDispFldRMark((short)LittleEndian.GetShort(varParam, 1));
               newCHP.SetDttmDispFldRMark(LittleEndian.GetInt(varParam, 3));
               System.Array.Copy(varParam, 7, xstDispFldRMark, 0, 32);
               newCHP.SetXstDispFldRMark(xstDispFldRMark);
               break;
          case 0x63:
               newCHP.SetIbstRMarkDel((short)param);
               break;
          case 0x64:
               short[] dttmRMarkDel = new short[2];
               dttmRMarkDel[0] = LittleEndian.GetShort(grpprl, offset - 4);
               dttmRMarkDel[1] = LittleEndian.GetShort(grpprl, offset - 2);
               newCHP.SetDttmRMarkDel(dttmRMarkDel);
               break;
          case 0x65:
               short[] brc = new short[2];
               brc[0] = (short)LittleEndian.GetShort(grpprl, offset - 4);
               brc[1] = (short)LittleEndian.GetShort(grpprl, offset - 2);
               newCHP.SetBrc(brc);
               break;
          case 0x66:
               newCHP.SetShd((short)param);
               break;
          case 0x67:
               break;
          case 0x68:
               break;
          case 0x69:
               break;
          case 0x6a:
               break;
          case 0x6b:
               break;
          case 0x6c:
               break;
          case 0x6d:
               newCHP.SetLidDefault((short)param);
               break;
          case 0x6e:
               newCHP.SetLidFE((short)param);
               break;
          case 0x6f:
               newCHP.SetIdctHint((byte)param);
               break;
      }
  }

        /**
         * Used to uncompress a property stored in a grpprl. These include
         * CharacterProperties, ParagraphProperties, TableProperties, and
         * SectionProperties.
         *
         * @param grpprl The compressed form of the property.
         * @param parent The base property of the property.
         * @param styleSheet The document's stylesheet.
         *
         * @return An object that should be casted to the appropriate property.
         */
        public static Object uncompressProperty(byte[] grpprl, Object parent, StyleSheet styleSheet)
        {
            return uncompressProperty(grpprl, parent, styleSheet, true);
        }

        /**
         * Used to uncompress a property stored in a grpprl. These include
         * CharacterProperties, ParagraphProperties, TableProperties, and
         * SectionProperties.
         *
         * @param grpprl The compressed form of the property.
         * @param parent The base property of the property.
         * @param styleSheet The document's stylesheet.
         *
         * @return An object that should be casted to the appropriate property.
         */
        public static Object uncompressProperty(byte[] grpprl, Object parent, StyleSheet styleSheet, bool doIstd)
        {
            Object newProperty = null;
            int offset = 0;
            int propertyType = PAP_TYPE;


            if (parent is ParagraphProperties)
            {
                try
                {
                    newProperty = ((ParagraphProperties)parent).Clone();
                }
                catch (Exception e) { }
                if (doIstd)
                {
                    ((ParagraphProperties)newProperty).SetIstd(LittleEndian.GetShort(grpprl, 0));

                    offset = 2;
                }
            }
            else if (parent is CharacterProperties)
            {
                try
                {
                    newProperty = ((CharacterProperties)parent).Clone();
                    ((CharacterProperties)newProperty).SetBaseIstd(((CharacterProperties)parent).GetIstd());
                }
                catch (Exception e) { }
                propertyType = CHP_TYPE;
            }
            else if (parent is SectionProperties)
            {
                newProperty = parent;
                propertyType = SEP_TYPE;
            }
            else if (parent is TableProperties)
            {
                newProperty = parent;
                propertyType = TAP_TYPE;
                offset = 2;//because this Is really just a papx
            }
            else
            {
                return null;
            }

            while (offset < grpprl.Length)
            {
                short sprm = LittleEndian.GetShort(grpprl, offset);
                offset += 2;

                byte spra = (byte)((sprm & 0xe000) >> 13);
                int opSize = 0;
                int param = 0;
                byte[] varParam = null;

                switch (spra)
                {
                    case 0:
                    case 1:
                        opSize = 1;
                        param = grpprl[offset];
                        break;
                    case 2:
                        opSize = 2;
                        param = LittleEndian.GetShort(grpprl, offset);
                        break;
                    case 3:
                        opSize = 4;
                        param = LittleEndian.GetInt(grpprl, offset);
                        break;
                    case 4:
                    case 5:
                        opSize = 2;
                        param = LittleEndian.GetShort(grpprl, offset);
                        break;
                    case 6://variable Size

                        //there Is one sprm that Is a very special case
                        if (sprm != unchecked((short)0xd608))
                        {
                            opSize = LittleEndian.GetUByte(grpprl, offset);
                            offset++;
                        }
                        else
                        {
                            opSize = LittleEndian.GetShort(grpprl, offset) - 1;
                            offset += 2;
                        }
                        varParam = new byte[opSize];
                        System.Array.Copy(grpprl, offset, varParam, 0, opSize);

                        break;
                    case 7:
                        opSize = 3;
                        byte[] threeByteInt = new byte[4];
                        threeByteInt[0] = grpprl[offset];
                        threeByteInt[1] = grpprl[offset + 1];
                        threeByteInt[2] = grpprl[offset + 2];
                        threeByteInt[3] = (byte)0;
                        param = LittleEndian.GetInt(threeByteInt, 0);
                        break;
                    default:
                        throw new Exception("unrecognized pap opcode");
                }

                offset += opSize;
                short operand = (short)(sprm & 0x1ff);
                byte type = (byte)((sprm & 0x1c00) >> 10);
                switch (propertyType)
                {
                    case PAP_TYPE:
                        if (type == 1)//papx stores TAP sprms along with PAP sprms
                        {
                            doPAPOperation((ParagraphProperties)newProperty, operand,
                                           param, varParam, grpprl,
                                           offset, spra);
                        }
                        break;
                    case CHP_TYPE:

                        DoCHPOperation((CharacterProperties)parent,
                                       (CharacterProperties)newProperty,
                                       operand, param, varParam,
                                       grpprl, offset, styleSheet);
                        break;
                    case SEP_TYPE:

                        doSEPOperation((SectionProperties)newProperty, operand, param, varParam);
                        break;
                    case TAP_TYPE:
                        if (type == 5)
                        {
                            doTAPOperation((TableProperties)newProperty, operand, param, varParam);
                        }
                        break;
                }


            }
            return newProperty;

        }
        /**
         * Performs an operation on a ParagraphProperties object. Used to uncompress
         * from a papx.
         *
         * @param newPAP The ParagraphProperties object to perform the operation on.
         * @param operand The operand that defines the operation.
         * @param param The operation's parameter.
         * @param varParam The operation's variable Length parameter.
         * @param grpprl The original papx.
         * @param offset The current offset in the papx.
         * @param spra A part of the sprm that defined this operation.
         */
        static void doPAPOperation(ParagraphProperties newPAP, int operand, int param,
                                   byte[] varParam, byte[] grpprl, int offset,
                                   int spra)
        {
            switch (operand)
            {
                case 0:
                    newPAP.SetIstd(param);
                    break;
                case 0x1:
                    //permuteIstd(newPAP, varParam);
                    break;
                case 0x2:
                    if (newPAP.GetIstd() <= 9 || newPAP.GetIstd() >= 1)
                    {
                        newPAP.SetIstd(newPAP.GetIstd() + param);
                        if (param > 0)
                        {
                            newPAP.SetIstd(Math.Max(newPAP.GetIstd(), 9));
                        }
                        else
                        {
                            newPAP.SetIstd(Math.Min(newPAP.GetIstd(), 1));
                        }
                    }
                    break;
                case 0x3:
                    newPAP.SetJc((byte)param);
                    break;
                case 0x4:
                    newPAP.SetFSideBySide((byte)param);
                    break;
                case 0x5:
                    newPAP.SetFKeep((byte)param);
                    break;
                case 0x6:
                    newPAP.SetFKeepFollow((byte)param);
                    break;
                case 0x7:
                    newPAP.SetFPageBreakBefore((byte)param);
                    break;
                case 0x8:
                    newPAP.SetBrcl((byte)param);
                    break;
                case 0x9:
                    newPAP.SetBrcp((byte)param);
                    break;
                case 0xa:
                    newPAP.SetIlvl((byte)param);
                    break;
                case 0xb:
                    newPAP.SetIlfo(param);
                    break;
                case 0xc:
                    newPAP.SetFNoLnn((byte)param);
                    break;
                case 0xd:
                    /**@todo handle tabs*/
                    break;
                case 0xe:
                    newPAP.SetDxaRight(param);
                    break;
                case 0xf:
                    newPAP.SetDxaLeft(param);
                    break;
                case 0x10:
                    newPAP.SetDxaLeft(newPAP.GetDxaLeft() + param);
                    newPAP.SetDxaLeft(Math.Max(0, newPAP.GetDxaLeft()));
                    break;
                case 0x11:
                    newPAP.SetDxaLeft1(param);
                    break;
                case 0x12:
                    short[] lspd = newPAP.GetLspd();
                    lspd[0] = LittleEndian.GetShort(grpprl, offset - 4);
                    lspd[1] = LittleEndian.GetShort(grpprl, offset - 2);
                    break;
                case 0x13:
                    newPAP.SetDyaBefore(param);
                    break;
                case 0x14:
                    newPAP.SetDyaAfter(param);
                    break;
                case 0x15:
                    /**@todo handle tabs*/
                    break;
                case 0x16:
                    newPAP.SetFInTable((byte)param);
                    break;
                case 0x17:
                    newPAP.SetFTtp((byte)param);
                    break;
                case 0x18:
                    newPAP.SetDxaAbs(param);
                    break;
                case 0x19:
                    newPAP.SetDyaAbs(param);
                    break;
                case 0x1a:
                    newPAP.SetDxaWidth(param);
                    break;
                case 0x1b:
                    /** @todo handle paragraph postioning*/
                    /*byte pcVert = (param & 0x0c) >> 2;
                    byte pcHorz = param & 0x03;
                    if(pcVert != 3)
                    {
                       newPAP._pcVert = pcVert;
                    }
                    if(pcHorz != 3)
                    {
                       newPAP._pcHorz = pcHorz;
                    }*/
                    break;
                case 0x1c:
                    //newPAP.SetBrcTop1((short)param);
                    break;
                case 0x1d:
                    //newPAP.SetBrcLeft1((short)param);
                    break;
                case 0x1e:
                    //newPAP.SetBrcBottom1((short)param);
                    break;
                case 0x1f:
                    //newPAP.SetBrcRight1((short)param);
                    break;
                case 0x20:
                    //newPAP.SetBrcBetween1((short)param);
                    break;
                case 0x21:
                    //newPAP.SetBrcBar1((byte)param);
                    break;
                case 0x22:
                    newPAP.SetDxaFromText(param);
                    break;
                case 0x23:
                    newPAP.SetWr((byte)param);
                    break;
                case 0x24:
                    short[] brcTop = newPAP.GetBrcTop();
                    brcTop[0] = (short)LittleEndian.GetShort(grpprl, offset - 4);
                    brcTop[1] = (short)LittleEndian.GetShort(grpprl, offset - 2);
                    break;
                case 0x25:
                    short[] brcLeft = newPAP.GetBrcLeft();
                    brcLeft[0] = (short)LittleEndian.GetShort(grpprl, offset - 4);
                    brcLeft[1] = (short)LittleEndian.GetShort(grpprl, offset - 2);
                    break;
                case 0x26:
                    short[] brcBottom = newPAP.GetBrcBottom();
                    brcBottom[0] = (short)LittleEndian.GetShort(grpprl, offset - 4);
                    brcBottom[1] = (short)LittleEndian.GetShort(grpprl, offset - 2);
                    break;
                case 0x27:
                    short[] brcRight = newPAP.GetBrcRight();
                    brcRight[0] = (short)LittleEndian.GetShort(grpprl, offset - 4);
                    brcRight[1] = (short)LittleEndian.GetShort(grpprl, offset - 2);
                    break;
                case 0x28:
                    short[] brcBetween = newPAP.GetBrcBetween();
                    brcBetween[0] = (short)LittleEndian.GetShort(grpprl, offset - 4);
                    brcBetween[1] = (short)LittleEndian.GetShort(grpprl, offset - 2);
                    break;
                case 0x29:
                    short[] brcBar = newPAP.GetBrcBar();
                    brcBar[0] = (short)LittleEndian.GetShort(grpprl, offset - 4);
                    brcBar[1] = (short)LittleEndian.GetShort(grpprl, offset - 2);
                    break;
                case 0x2a:
                    newPAP.SetFNoAutoHyph((byte)param);
                    break;
                case 0x2b:
                    newPAP.SetDyaHeight(param);
                    break;
                case 0x2c:
                    newPAP.SetDcs((short)param);
                    break;
                case 0x2d:
                    newPAP.SetShd((short)param);
                    break;
                case 0x2e:
                    newPAP.SetDyaFromText(param);
                    break;
                case 0x2f:
                    newPAP.SetDxaFromText(param);
                    break;
                case 0x30:
                    newPAP.SetFLocked((byte)param);
                    break;
                case 0x31:
                    newPAP.SetFWidowControl((byte)param);
                    break;
                case 0x32:
                    //undocumented
                    break;
                case 0x33:
                    newPAP.SetFKinsoku((byte)param);
                    break;
                case 0x34:
                    newPAP.SetFWordWrap((byte)param);
                    break;
                case 0x35:
                    newPAP.SetFOverflowPunct((byte)param);
                    break;
                case 0x36:
                    newPAP.SetFTopLinePunct((byte)param);
                    break;
                case 0x37:
                    newPAP.SetFAutoSpaceDE((byte)param);
                    break;
                case 0x38:
                    newPAP.SetFAutoSpaceDN((byte)param);
                    break;
                case 0x39:
                    newPAP.SetWAlignFont(param);
                    break;
                case 0x3a:
                    newPAP.SetFontAlign((short)param);
                    break;
                case 0x3b:
                    //obsolete
                    break;
                case 0x3e:
                    newPAP.SetAnld(varParam);
                    break;
                case 0x3f:
                    //don't really need this. spec Is confusing regarding this
                    //sprm
                    break;
                case 0x40:
                    //newPAP._lvl = param;
                    break;
                case 0x41:
                    //?
                    break;
                case 0x43:
                    //?
                    break;
                case 0x44:
                    //?
                    break;
                case 0x45:
                    if (spra == 6)
                    {
                        newPAP.SetNumrm(varParam);
                    }
                    else
                    {
                        /**@todo handle large PAPX from data stream*/
                    }
                    break;

                case 0x47:
                    newPAP.SetFUsePgsuSettings((byte)param);
                    break;
                case 0x48:
                    newPAP.SetFAdjustRight((byte)param);
                    break;
                default:
                    break;
            }
        }
        /**
         * Used to uncompress a table property. Performs an operation defined
         * by a sprm stored in a tapx.
         *
         * @param newTAP The TableProperties object to perform the operation on.
         * @param operand The operand that defines this operation.
         * @param param The parameter for this operation.
         * @param varParam Variable Length parameter for this operation.
         */
        static void doTAPOperation(TableProperties newTAP, int operand, int param, byte[] varParam)
        {
            switch (operand)
            {
                case 0:
                    newTAP.SetJc((short)param);
                    break;
                case 0x01:
                    {
                        short[] rgdxaCenter = newTAP.GetRgdxaCenter();
                        short itcMac = newTAP.GetItcMac();
                        int adjust = param - (rgdxaCenter[0] + newTAP.GetDxaGapHalf());
                        for (int x = 0; x < itcMac; x++)
                        {
                            rgdxaCenter[x] += (short)adjust;
                        }
                        break;
                    }
                case 0x02:
                    {
                        short[] rgdxaCenter = newTAP.GetRgdxaCenter();
                        if (rgdxaCenter != null)
                        {
                            int adjust = newTAP.GetDxaGapHalf() - param;
                            rgdxaCenter[0] += (short)adjust;
                        }
                        newTAP.SetDxaGapHalf(param);
                        break;
                    }
                case 0x03:
                    newTAP.SetFCantSplit(GetFlag(param));
                    break;
                case 0x04:
                    newTAP.SetFTableHeader(GetFlag(param));
                    break;
                case 0x05:
                    {
                        short[] brcTop = newTAP.GetBrcTop();
                        short[] brcLeft = newTAP.GetBrcLeft();
                        short[] brcBottom = newTAP.GetBrcBottom();
                        short[] brcRight = newTAP.GetBrcRight();
                        short[] brcVertical = newTAP.GetBrcVertical();
                        short[] brcHorizontal = newTAP.GetBrcHorizontal();

                        brcTop[0] = LittleEndian.GetShort(varParam, 0);
                        brcTop[1] = LittleEndian.GetShort(varParam, 2);

                        brcLeft[0] = LittleEndian.GetShort(varParam, 4);
                        brcLeft[1] = LittleEndian.GetShort(varParam, 6);

                        brcBottom[0] = LittleEndian.GetShort(varParam, 8);
                        brcBottom[1] = LittleEndian.GetShort(varParam, 10);

                        brcRight[0] = LittleEndian.GetShort(varParam, 12);
                        brcRight[1] = LittleEndian.GetShort(varParam, 14);

                        brcHorizontal[0] = LittleEndian.GetShort(varParam, 16);
                        brcHorizontal[1] = LittleEndian.GetShort(varParam, 18);

                        brcVertical[0] = LittleEndian.GetShort(varParam, 20);
                        brcVertical[1] = LittleEndian.GetShort(varParam, 22);
                        break;
                    }
                case 0x06:
                    //obsolete, used in word 1.x
                    break;
                case 0x07:
                    newTAP.SetDyaRowHeight(param);
                    break;
                case 0x08:
                    {
                        short[] rgdxaCenter = new short[varParam[0] + 1];
                        TableCellDescriptor[] rgtc = new TableCellDescriptor[varParam[0]];
                        short itcMac = varParam[0];
                        //I use varParam[0] and newTAP._itcMac interchangably
                        newTAP.SetItcMac(itcMac);
                        newTAP.SetRgdxaCenter(rgdxaCenter);
                        newTAP.SetRgtc(rgtc);

                        for (int x = 0; x < itcMac; x++)
                        {
                            rgdxaCenter[x] = LittleEndian.GetShort(varParam, 1 + (x * 2));
                            rgtc[x] = TableCellDescriptor.ConvertBytesToTC(varParam, 1 + ((itcMac + 1) * 2) + (x * 20));
                        }
                        rgdxaCenter[itcMac] = LittleEndian.GetShort(varParam, 1 + (itcMac * 2));
                        break;
                    }
                case 0x09:
                    /** @todo handle cell shading*/
                    break;
                case 0x0a:
                    /** @todo handle word defined table styles*/
                    break;
                case 0x20:
                    {
                        TCAbstractType[] rgtc = newTAP.GetRgtc();

                        for (int x = varParam[0]; x < varParam[1]; x++)
                        {

                            if ((varParam[2] & 0x08) > 0)
                            {
                                short[] brcRight = rgtc[x].GetBrcRight();
                                brcRight[0] = LittleEndian.GetShort(varParam, 6);
                                brcRight[1] = LittleEndian.GetShort(varParam, 8);
                            }
                            else if ((varParam[2] & 0x04) > 0)
                            {
                                short[] brcBottom = rgtc[x].GetBrcBottom();
                                brcBottom[0] = LittleEndian.GetShort(varParam, 6);
                                brcBottom[1] = LittleEndian.GetShort(varParam, 8);
                            }
                            else if ((varParam[2] & 0x02) > 0)
                            {
                                short[] brcLeft = rgtc[x].GetBrcLeft();
                                brcLeft[0] = LittleEndian.GetShort(varParam, 6);
                                brcLeft[1] = LittleEndian.GetShort(varParam, 8);
                            }
                            else if ((varParam[2] & 0x01) > 0)
                            {
                                short[] brcTop = rgtc[x].GetBrcTop();
                                brcTop[0] = LittleEndian.GetShort(varParam, 6);
                                brcTop[1] = LittleEndian.GetShort(varParam, 8);
                            }
                        }
                        break;
                    }
                case 0x21:
                    int index = (param & unchecked((int)0xff000000)) >> 24;
                    int count = (param & 0x00ff0000) >> 16;
                    int width = (param & 0x0000ffff);
                    int itcMac1 = newTAP.GetItcMac();

                    short[] rgdxaCenter1 = new short[itcMac1 + count + 1];
                    TableCellDescriptor[] rgtc1 = new TableCellDescriptor[itcMac1 + count];
                    if (index >= itcMac1)
                    {
                        index = itcMac1;
                        System.Array.Copy(newTAP.GetRgdxaCenter(), 0, rgdxaCenter1, 0, itcMac1 + 1);
                        System.Array.Copy(newTAP.GetRgtc(), 0, rgtc1, 0, itcMac1);
                    }
                    else
                    {
                        //copy rgdxaCenter
                        System.Array.Copy(newTAP.GetRgdxaCenter(), 0, rgdxaCenter1, 0, index + 1);
                        System.Array.Copy(newTAP.GetRgdxaCenter(), index + 1, rgdxaCenter1, index + count, itcMac1 - (index));
                        //copy rgtc
                        System.Array.Copy(newTAP.GetRgtc(), 0, rgtc1, 0, index);
                        System.Array.Copy(newTAP.GetRgtc(), index, rgtc1, index + count, itcMac1 - index);
                    }

                    for (int x = index; x < index + count; x++)
                    {
                        rgtc1[x] = new TableCellDescriptor();
                        rgdxaCenter1[x] = (short)(rgdxaCenter1[x - 1] + width);
                    }
                    rgdxaCenter1[index + count] = (short)(rgdxaCenter1[(index + count) - 1] + width);
                    break;
                /**@todo handle table sprms from complex files*/
                case 0x22:
                case 0x23:
                case 0x24:
                case 0x25:
                case 0x26:
                case 0x27:
                case 0x28:
                case 0x29:
                case 0x2a:
                case 0x2b:
                case 0x2c:
                    break;
                default:
                    break;
            }
        }
        /**
         * Used in decompression of a sepx. This performs an operation defined by
         * a single sprm.
         *
         * @param newSEP The SectionProperty to perfrom the operation on.
         * @param operand The operation to perform.
         * @param param The operation's parameter.
         * @param varParam The operation variable Length parameter.
         */
        static void doSEPOperation(SectionProperties newSEP, int operand, int param, byte[] varParam)
        {
            switch (operand)
            {
                case 0:
                    newSEP.SetCnsPgn((byte)param);
                    break;
                case 0x1:
                    newSEP.SetIHeadingPgn((byte)param);
                    break;
                case 0x2:
                    newSEP.SetOlstAnm(varParam);
                    break;
                case 0x3:
                    //not quite sure
                    break;
                case 0x4:
                    //not quite sure
                    break;
                case 0x5:
                    newSEP.SetFEvenlySpaced(GetFlag(param));
                    break;
                case 0x6:
                    newSEP.SetFUnlocked(GetFlag(param));
                    break;
                case 0x7:
                    newSEP.SetDmBinFirst((short)param);
                    break;
                case 0x8:
                    newSEP.SetDmBinOther((short)param);
                    break;
                case 0x9:
                    newSEP.SetBkc((byte)param);
                    break;
                case 0xa:
                    newSEP.SetFTitlePage(GetFlag(param));
                    break;
                case 0xb:
                    newSEP.SetCcolM1((short)param);
                    break;
                case 0xc:
                    newSEP.SetDxaColumns(param);
                    break;
                case 0xd:
                    newSEP.SetFAutoPgn(GetFlag(param));
                    break;
                case 0xe:
                    newSEP.SetNfcPgn((byte)param);
                    break;
                case 0xf:
                    newSEP.SetDyaPgn((short)param);
                    break;
                case 0x10:
                    newSEP.SetDxaPgn((short)param);
                    break;
                case 0x11:
                    newSEP.SetFPgnRestart(GetFlag(param));
                    break;
                case 0x12:
                    newSEP.SetFEndNote(GetFlag(param));
                    break;
                case 0x13:
                    newSEP.SetLnc((byte)param);
                    break;
                case 0x14:
                    newSEP.SetGrpfIhdt((byte)param);
                    break;
                case 0x15:
                    newSEP.SetNLnnMod((short)param);
                    break;
                case 0x16:
                    newSEP.SetDxaLnn(param);
                    break;
                case 0x17:
                    newSEP.SetDyaHdrTop(param);
                    break;
                case 0x18:
                    newSEP.SetDyaHdrBottom(param);
                    break;
                case 0x19:
                    newSEP.SetFLBetween(GetFlag(param));
                    break;
                case 0x1a:
                    newSEP.SetVjc((byte)param);
                    break;
                case 0x1b:
                    newSEP.SetLnnMin((short)param);
                    break;
                case 0x1c:
                    newSEP.SetPgnStart((short)param);
                    break;
                case 0x1d:
                    newSEP.SetDmOrientPage((byte)param);
                    break;
                case 0x1e:
                    //nothing
                    break;
                case 0x1f:
                    newSEP.SetXaPage(param);
                    break;
                case 0x20:
                    newSEP.SetYaPage(param);
                    break;
                case 0x21:
                    newSEP.SetDxaLeft(param);
                    break;
                case 0x22:
                    newSEP.SetDxaRight(param);
                    break;
                case 0x23:
                    newSEP.SetDyaTop(param);
                    break;
                case 0x24:
                    newSEP.SetDyaBottom(param);
                    break;
                case 0x25:
                    newSEP.SetDzaGutter(param);
                    break;
                case 0x26:
                    newSEP.SetDmPaperReq((short)param);
                    break;
                case 0x27:
                    newSEP.SetFPropMark(GetFlag(varParam[0]));
                    break;
                case 0x28:
                    break;
                case 0x29:
                    break;
                case 0x2a:
                    break;
                case 0x2b:
                    short[] brcTop = newSEP.GetBrcTop();
                    brcTop[0] = (short)(param & 0xffff);
                    brcTop[1] = (short)((param & 0xffff0000) >> 16);
                    break;
                case 0x2c:
                    short[] brcLeft = newSEP.GetBrcLeft();
                    brcLeft[0] = (short)(param & 0xffff);
                    brcLeft[1] = (short)((param & 0xffff0000) >> 16);
                    break;
                case 0x2d:
                    short[] brcBottom = newSEP.GetBrcBottom();
                    brcBottom[0] = (short)(param & 0xffff);
                    brcBottom[1] = (short)((param & 0xffff0000) >> 16);
                    break;
                case 0x2e:
                    short[] brcRight = newSEP.GetBrcRight();
                    brcRight[0] = (short)(param & 0xffff);
                    brcRight[1] = (short)((param & 0xffff0000) >> 16);
                    break;
                case 0x2f:
                    newSEP.SetPgbProp(param);
                    break;
                case 0x30:
                    newSEP.SetDxtCharSpace(param);
                    break;
                case 0x31:
                    newSEP.SetDyaLinePitch(param);
                    break;
                case 0x33:
                    newSEP.SetWTextFlow((short)param);
                    break;
                default:
                    break;
            }

        }
        /**
         * Converts an byte value into a boolean. The byte parameter can be 1,0, 128,
         * or 129. if it Is 128, this function returns the same value as oldVal. If
         * it Is 129, this function returns !oldVal. This Is used for certain sprms
         *
         * @param x The byte value to convert.
         * @param oldVal The old bool value.
         *
         * @return A bool whose value depends on x and oldVal.
         */
        private static bool GetCHPFlag(byte x, bool oldVal)
        {
            switch (x)
            {
                case 0:
                    return false;
                case 1:
                    return true;
                case (byte)0x80:
                    return oldVal;
                case (byte)0x81:
                    return !oldVal;
                default:
                    return false;
            }
        }

        /**
         * Converts an int into a boolean. If the int Is non-zero, it returns true.
         * Otherwise it returns false.
         *
         * @param x The int to convert.
         *
         * @return A bool whose value depends on x.
         */
        public static bool GetFlag(int x)
        {
            if (x != 0)
            {
                return true;
            }
            else
            {
                return false;
            }
        }
    }
}